"""
Allegro — weryfikator statusów zwrotów
======================================
Skrypt sprawdza zamówienia ze zwrotami i porównuje ich status
oraz kwoty z rzeczywistymi danymi z Allegro API.

Wejście: eksport CSV z Sellasist (bez przygotowywania — wrzucasz plik 1:1)
Wyjście: wynik_zwroty_YYYYMMDD_HHMM.csv + wszystkie_zwroty_YYYYMMDD_HHMM.csv

Wymagania:
    pip install requests

Uruchomienie:
    python allegro_zwroty.py
"""

import requests
import json
import time
import csv
import subprocess
import webbrowser
from datetime import datetime
from getpass import getpass
from collections import defaultdict


# ─── KONFIGURACJA ────────────────────────────────────────────────────────────

ALLEGRO_API = "https://api.allegro.pl"
AUTH_URL    = "https://allegro.pl/auth/oauth"

# Nazwy kolumn w eksporcie Sellasist
COL_ID       = "ID"
COL_ORDER_ID = "Zewnętrzny nr zamówienia"
COL_WARTOSC  = "Wartość"
COL_WALUTA   = "Waluta"

# Status wpisywany zawsze do wynikowego CSV
FIXED_STATUS = "ZWROT ZREALIZOWANY"

# Separator i kodowanie eksportu Sellasist
CSV_DELIMITER = ","
CSV_ENCODING  = "utf-8-sig"

# DEBUG — wypisuje surowy JSON pierwszego zwrotu z API
DEBUG_FIRST = True


# ─── AUTORYZACJA (Device Code Flow) ──────────────────────────────────────────

def open_browser(url: str):
    """Otwiera przeglądarkę — działa na Windows, Mac i Linux."""
    try:
        webbrowser.open(url)
        return
    except Exception:
        pass
    try:
        subprocess.Popen(["cmd", "/c", "start", "", url], shell=False)
        return
    except Exception:
        pass
    try:
        subprocess.Popen(["xdg-open", url])
    except Exception:
        pass


def authorize(client_id: str, client_secret: str) -> str:
    r = requests.post(
        f"{AUTH_URL}/device",
        auth=(client_id, client_secret),
        data={"client_id": client_id},
    )
    if r.status_code != 200:
        print(f"❌ Błąd autoryzacji: {r.status_code} — {r.text}")
        raise SystemExit(1)

    data       = r.json()
    code       = data["device_code"]
    user_code  = data["user_code"]
    verify_url = data["verification_uri_complete"]
    interval   = data.get("interval", 5)

    print(f"\n{'='*60}")
    print(f"  Zaloguj się w Allegro pod adresem:")
    print(f"  {verify_url}")
    print(f"\n  Albo wejdź na: {data['verification_uri']}")
    print(f"  i wpisz kod: {user_code}")
    print(f"{'='*60}\n")

    open_browser(verify_url)
    print("Czekam na autoryzację... (zaloguj się w oknie przeglądarki)\n")

    token_url = f"{AUTH_URL}/token"
    while True:
        time.sleep(interval)
        tr = requests.post(
            token_url,
            auth=(client_id, client_secret),
            data={
                "grant_type":  "urn:ietf:params:oauth:grant-type:device_code",
                "device_code": code,
            },
        )
        if tr.status_code == 200:
            token = tr.json()["access_token"]
            print("✅ Autoryzacja zakończona sukcesem!\n")
            return token

        err = tr.json().get("error", "")
        if err == "authorization_pending":
            print("  ...czekam na kliknięcie w przeglądarce...")
        elif err == "slow_down":
            interval += 2
        else:
            print(f"❌ Błąd autoryzacji: {tr.json()}")
            raise SystemExit(1)


# ─── ALLEGRO API ──────────────────────────────────────────────────────────────

def api_get(token: str, path: str, params: dict = None) -> dict:
    headers = {
        "Authorization": f"Bearer {token}",
        "Accept": "application/vnd.allegro.public.v1+json",
    }
    r = requests.get(f"{ALLEGRO_API}{path}", headers=headers, params=params)
    if r.status_code == 429:
        print("  ⏳ Rate limit — czekam 15s...")
        time.sleep(15)
        return api_get(token, path, params)
    r.raise_for_status()
    return r.json()


# ─── POBIERANIE WSZYSTKICH ZWROTÓW (jednorazowo) ─────────────────────────────

def fetch_all_refunds(token: str) -> list:
    """Pobiera WSZYSTKIE zwroty z konta przez paginację offset/limit."""
    all_refunds = []
    offset = 0
    limit  = 100
    page   = 1

    print("Pobieram wszystkie zwroty z Allegro API...")

    while True:
        try:
            data = api_get(token, "/payments/refunds", params={"limit": limit, "offset": offset})
        except Exception as e:
            print(f"  ⚠️  Błąd /payments/refunds (strona {page}, offset {offset}): {e}")
            break

        batch = data.get("refunds", [])
        all_refunds.extend(batch)
        print(f"  Strona {page} — pobrano {len(batch)} zwrotów (łącznie: {len(all_refunds)})")

        if len(batch) < limit:
            break

        offset += limit
        page   += 1
        time.sleep(0.3)

    print(f"✅ Pobrano łącznie {len(all_refunds)} zwrotów.\n")
    return all_refunds


def build_refunds_index(all_refunds: list) -> tuple:
    """Buduje słowniki {payment_id: [zwroty]} i {order_id: [zwroty]}."""
    by_payment_id = defaultdict(list)
    by_order_id   = defaultdict(list)

    for r in all_refunds:
        pid = (r.get("payment") or {}).get("id")
        oid = (r.get("order")   or {}).get("id")
        if pid:
            by_payment_id[pid].append(r)
        if oid:
            by_order_id[oid].append(r)

    return dict(by_payment_id), dict(by_order_id)


def save_all_refunds_to_csv(all_refunds: list, filepath: str):
    """Zapisuje pełną listę zwrotów do CSV."""
    if not all_refunds:
        print("  Brak zwrotów do zapisania.")
        return

    fieldnames = [
        "refund_id", "payment_id", "order_id",
        "total_amount", "currency",
        "delivery_cost", "occurred_at", "status", "reason",
    ]

    with open(filepath, "w", encoding="utf-8-sig", newline="") as f:
        writer = csv.DictWriter(f, fieldnames=fieldnames, delimiter=";", extrasaction="ignore")
        writer.writeheader()

        for r in all_refunds:
            payment   = r.get("payment") or {}
            order     = r.get("order")   or {}
            total_val = r.get("totalValue") or r.get("totalAmount") or r.get("amount") or {}

            if isinstance(total_val, dict):
                amount   = total_val.get("amount", "")
                currency = total_val.get("currency", "PLN")
            else:
                amount, currency = total_val, "PLN"

            delivery        = r.get("deliveryCost") or r.get("shippingCost") or {}
            delivery_amount = delivery.get("amount", "") if isinstance(delivery, dict) else (delivery or "")

            writer.writerow({
                "refund_id":     r.get("id", ""),
                "payment_id":    payment.get("id", ""),
                "order_id":      order.get("id", ""),
                "total_amount":  amount,
                "currency":      currency,
                "delivery_cost": delivery_amount,
                "occurred_at":   r.get("occurredAt") or r.get("createdAt") or "",
                "status":        r.get("status", ""),
                "reason":        r.get("reason") or r.get("reasonType") or "",
            })

    print(f"  Lista wszystkich zwrotów zapisana: {filepath}\n")


# ─── HELPERY ──────────────────────────────────────────────────────────────────

def _get_nested(obj: dict, *paths: str) -> float:
    for path in paths:
        val = obj
        for key in path.split("."):
            val = val.get(key) if isinstance(val, dict) else None
        if val is None:
            continue
        if isinstance(val, (int, float)):
            return float(val)
        if isinstance(val, str):
            try:
                return float(val.replace(",", "."))
            except ValueError:
                pass
        if isinstance(val, dict):
            amt = val.get("amount")
            if amt is not None:
                try:
                    return float(amt)
                except (ValueError, TypeError):
                    pass
    return 0.0


def _get_refund_datetime(refund: dict) -> str:
    for path in ["occurredAt", "createdAt", "date", "refundedAt", "updatedAt"]:
        val = refund.get(path)
        if val and isinstance(val, str):
            try:
                dt = datetime.fromisoformat(val.replace("Z", "+00:00"))
                return dt.strftime("%Y-%m-%d %H:%M")
            except Exception:
                return val
    return ""


# ─── ANALIZA ZWROTU ───────────────────────────────────────────────────────────

def analyze_refund(order_total: float, refunds: list) -> dict:
    """Porównuje kwotę z Sellasist ze zwrotami z Allegro."""
    refunded_total = 0.0
    last_dt        = ""

    for r in refunds:
        amt = _get_nested(r,
            "totalValue.amount",
            "totalAmount.amount", "totalAmount",
            "amount.amount",      "amount",
            "value.amount",       "value",
        )
        refunded_total += amt

        dt = _get_refund_datetime(r)
        if dt:
            last_dt = dt

    refund_count = len(refunds)
    missing      = round(order_total - refunded_total, 2)

    if refund_count == 0:
        status = "BRAK ZWROTU"
    elif missing > 0.01:
        status = "CZĘŚCIOWY"
    elif missing < -0.01:
        status = "NADPŁATA"
    else:
        status = "PEŁNY"

    if status == "BRAK ZWROTU":
        akcja = "→ PRZENIEŚ DO OCZEKUJĄCYCH"
    elif status == "CZĘŚCIOWY":
        akcja = f"→ SPRAWDŹ (brakuje {missing:.2f} PLN)"
    elif status == "NADPŁATA":
        akcja = f"→ SPRAWDŹ (nadpłata {abs(missing):.2f} PLN)"
    else:
        akcja = "✅ OK — PRZENIEŚ DO ZREALIZOWANYCH"

    return {
        "refunded_total": f"{refunded_total:.2f}",
        "refund_count":   refund_count,
        "refund_status":  status,
        "missing_amount": f"{missing:.2f}" if missing != 0 else "0.00",
        "refund_datetime": last_dt,
        "akcja":          akcja,
    }


# ─── WCZYTANIE EKSPORTU SELLASIST ────────────────────────────────────────────

def load_orders_from_sellasist(filepath: str) -> list:
    """Wczytuje eksport z Sellasist 1:1 — bez przygotowywania pliku."""
    orders = []

    with open(filepath, encoding=CSV_ENCODING, newline="") as f:
        reader = csv.DictReader(f, delimiter=CSV_DELIMITER)

        # Sprawdź wymagane kolumny
        required = [COL_ID, COL_ORDER_ID, COL_WARTOSC, COL_WALUTA]
        missing  = [c for c in required if c not in (reader.fieldnames or [])]
        if missing:
            print(f"❌ Brak kolumn w pliku: {missing}")
            print(f"   Dostępne kolumny: {list(reader.fieldnames)}")
            raise SystemExit(1)

        for row in reader:
            oid = row.get(COL_ORDER_ID, "").strip().strip('"')
            # Pomijaj wiersze bez UUID Allegro (zamówienia z innych źródeł / puste)
            if not oid or len(oid) < 30:
                continue

            wartosc_raw = row.get(COL_WARTOSC, "0").strip().strip('"').replace(",", ".")
            try:
                wartosc = float(wartosc_raw)
            except ValueError:
                wartosc = 0.0

            orders.append({
                "sellaasist_id": row.get(COL_ID, "").strip().strip('"'),
                "order_id":      oid,
                "wartosc":       wartosc,
                "waluta":        row.get(COL_WALUTA, "PLN").strip().strip('"'),
            })

    return orders


# ─── POBIERANIE PAYMENT_ID ────────────────────────────────────────────────────

def get_payment_id(token: str, order_id: str) -> str:
    """Pobiera payment.id z API Allegro dla danego zamówienia."""
    try:
        data    = api_get(token, f"/order/checkout-forms/{order_id}")
        payment = data.get("payment", {})
        if isinstance(payment, dict):
            return payment.get("id", "")
    except Exception as e:
        print(f"  ⚠️  Błąd pobierania zamówienia {order_id[:8]}...: {e}")
    return ""


# ─── GŁÓWNA LOGIKA ────────────────────────────────────────────────────────────

def main():
    print("\n" + "="*60)
    print("  ALLEGRO — Weryfikator zwrotów")
    print("="*60 + "\n")

    client_id     = input("Client ID aplikacji Allegro: ").strip()
    client_secret = getpass("Client Secret (ukryty): ").strip()

    print("\nPodaj ścieżkę do eksportu CSV z Sellasist.")
    print("Przykład: C:\\Users\\TwojeImie\\Desktop\\eksport_zamowien.csv")
    csv_path = input("Ścieżka do pliku: ").strip().strip('"')

    # Wczytaj zamówienia
    print(f"\nWczytuję zamówienia z: {csv_path}")
    try:
        orders = load_orders_from_sellasist(csv_path)
    except FileNotFoundError:
        print(f"❌ Nie znaleziono pliku: {csv_path}")
        raise SystemExit(1)

    print(f"Znaleziono {len(orders)} zamówień Allegro do sprawdzenia.\n")

    if not orders:
        print("❌ Brak zamówień do sprawdzenia.")
        print("   Upewnij się że plik zawiera zamówienia z Allegro")
        print(f"   (kolumna '{COL_ORDER_ID}' musi mieć UUID-y).")
        raise SystemExit(1)

    # Autoryzacja
    print("Uruchamiam autoryzację Allegro...")
    token = authorize(client_id, client_secret)

    # Jednorazowe pobranie wszystkich zwrotów
    all_refunds = fetch_all_refunds(token)

    timestamp    = datetime.now().strftime("%Y%m%d_%H%M")
    refunds_file = f"wszystkie_zwroty_{timestamp}.csv"
    output_file  = f"wynik_zwroty_{timestamp}.csv"

    save_all_refunds_to_csv(all_refunds, refunds_file)

    if DEBUG_FIRST and all_refunds:
        print("[DEBUG] Przykładowy zwrot (pierwszy z listy):")
        print(json.dumps(all_refunds[0], indent=2, ensure_ascii=False))
        print()

    by_payment_id, by_order_id = build_refunds_index(all_refunds)

    # Przetwarzanie zamówień
    fieldnames = [
        "sellaasist_id", "order_id", "status",
        "wartosc_sellasist", "waluta",
        "refunded_total", "refund_count", "refund_status",
        "missing_amount", "refund_datetime", "akcja",
    ]

    with open(output_file, "w", encoding="utf-8-sig", newline="") as f:
        writer = csv.DictWriter(f, fieldnames=fieldnames, delimiter=";")
        writer.writeheader()

        for i, order in enumerate(orders, 1):
            oid = order["order_id"]
            sid = order["sellaasist_id"]

            print(f"[{i}/{len(orders)}] Sellaasist: {sid} | Allegro: {oid[:8]}...")

            # Pobierz payment_id (1 request per zamówienie — niezbędne do powiązania)
            payment_id = get_payment_id(token, oid)

            # Szukaj zwrotów w indeksie lokalnie — zero dodatkowych requestów
            if payment_id and payment_id in by_payment_id:
                refunds = by_payment_id[payment_id]
            elif oid in by_order_id:
                refunds = by_order_id[oid]
            else:
                refunds = []

            analysis = analyze_refund(order_total=order["wartosc"], refunds=refunds)

            writer.writerow({
                "sellaasist_id":     sid,
                "order_id":          oid,
                "status":            FIXED_STATUS,
                "wartosc_sellasist": f"{order['wartosc']:.2f}",
                "waluta":            order["waluta"],
                "refunded_total":    analysis["refunded_total"],
                "refund_count":      analysis["refund_count"],
                "refund_status":     analysis["refund_status"],
                "missing_amount":    analysis["missing_amount"],
                "refund_datetime":   analysis["refund_datetime"],
                "akcja":             analysis["akcja"],
            })

            time.sleep(0.3)

    print(f"\n{'='*60}")
    print(f"✅ Gotowe!")
    print(f"   Wynik analizy:   {output_file}")
    print(f"   Lista zwrotów:   {refunds_file}")
    print(f"{'='*60}")
    print("\nLegenda AKCJA:")
    print("  ✅ OK                      → zwrot pełny")
    print("  → PRZENIEŚ DO OCZEKUJĄCYCH → brak zwrotu w Allegro")
    print("  → SPRAWDŹ (brakuje X PLN)  → zwrot częściowy")
    print("  → SPRAWDŹ (nadpłata X PLN) → zwrócono więcej niż wartość zamówienia")


if __name__ == "__main__":
    main()
